#!/usr/bin/awk -f
#
# Awk script to generate an list of available APs via the Linux WiFi driver.
#

function dbg(str) {
  if (debug != 0) {
    print "debug: " str;
  }
}

function convertAuthentication(field) {
  # Note the values returned here should match those
  # in wlan_sta_linux.html:id_network_auth
  if      (field == "OPEN"    ) { return "OPEN"    ; }
  else if (field == "WEP"     ) { return "SHARED"  ; }
  else if (field == "WPA-PSK" ) { return "WPA-PSK" ; }
  else if (field == "WPA-EAP" ) { return "WPA"     ; }
  else if (field == "WPA2-PSK") { return "WPA2-PSK"; }
  else if (field == "WPA2-EAP") { return "WPA2"    ; }
  else                          { return ""        ; }
}

function convertEncryption(field) {
  # Note the values returned here should match those
  # in wlan_sta_linux.html:id_wpa_encryption
  if      (field == "NONE") { return "NONE"; }
  else if (field == "CCMP") { return "AES" ; }
  else if (field == "TKIP") { return "TKIP"; }
  else                      { return ""    ; }
}

function freq2Channel(frequency) {
  if      (frequency == "2412") { return "1" ; }
  else if (frequency == "2417") { return "2" ; }
  else if (frequency == "2422") { return "3" ; }
  else if (frequency == "2427") { return "4" ; }
  else if (frequency == "2432") { return "5" ; }
  else if (frequency == "2437") { return "6" ; }
  else if (frequency == "2442") { return "7" ; }
  else if (frequency == "2447") { return "8" ; }
  else if (frequency == "2452") { return "9" ; }
  else if (frequency == "2457") { return "10"; }
  else if (frequency == "2462") { return "11"; }
  else if (frequency == "2467") { return "12"; }
  else if (frequency == "2472") { return "13"; }
  else if (frequency == "2484") { return "14"; }
  else                          { return ""  ; }
}

function sprintQuotedPair(name, value, suffix) {
  return sprintf(" \"%s\":\"%s\"%s", name, value, suffix);
}

function skipRecordSpace() {
  if (match(record, "^[[:space:]]+") != 0) {
    record = substr(record, RLENGTH +1);
  }
}

# Extract the setting matching the 'regexp' into the 'settings' array at the
# specified 'idx'. Remove the setting from the 'record' as well as any
# trailing space.
function getSetting(regexp, settings, idx, msg) {
  if (match(record, regexp) != 0) {
    settings[idx] = substr(record, RSTART, RLENGTH);
    record = substr(record, RLENGTH +1);
    skipRecordSpace();
    dbg(msg settings[idx]);
  }
  else {
    dbg("  " idx " match failed, ignoring");
    next;
  }
}

BEGIN {
  debug = 0;                    # 0/1 => o/p debug messages
  lines[];                      # store the lines to output
  linesIdx = 1;                  
  writeConnected = 0;           # 1 => write the connected field in the output
}

# Process wpa_cli scan output records
/^([[:xdigit:]]{2}:){5}[[:xdigit:]]{2}/ {
  dbg($0);

  # For processing the i/p record
  record = $0;

  # Get the mandatory fields
  #   Eg: 02:00:c9:87:38:c0 2437 -39
  # Note: Use an array so we can pass by reference to the function below.
  settings[];  # indexes: "bssid", "frequency", "signalLevel"
  {
    # Do this the long way since this awk doesn't support the groups
    # array: a[n,  "start"],
    getSetting("^([[:xdigit:]]{2}:){5}[[:xdigit:]]{2}",
               settings, "bssid", "  bssid         : ");
    getSetting("^[[:digit:]]+",
               settings, "frequency", "  frequency     : ");
    getSetting("-*[[:digit:]]+",
               settings, "signalLevel", "  signal level  : ");
  }

  # Get any flags and convert them to security options.
  #   Eg: [WPA2-EAP+PSK-CCMP][ESS]
  security[];
  secIdx = 1;
  gotSecurity = 0;
  isAp = 0;
  while (1) {

    # Find the next flag: [...]
    start = index(record, "[");
    end   = index(record, "]");
    if ((start == 0) || (end == 0)) {
      break;
    }
    flag = substr(record, start +1, end - start -1);
    dbg("  flag          : " flag);
    record = substr(record, end +1); # Remove the current flag

    # Detect ESS flag
    if (match(flag, "ESS") != 0) {
      isAp = 1;
      continue;
    }

    # Split all the flag fields and walk through them to work out the supported
    # security options for this flag.
    # Note: '+' denotes an extra authentication option
    authentications[];
    encryptions[];
    authIdx = 1;
    encIdx = 1;
    authBase = "";
    opts[];
    num = split(flag, opts, "[+-]");
    for (fieldIdx = 1; fieldIdx <= num; ++fieldIdx) {
      field = opts[fieldIdx];
      dbg("    field: " field);

      if ((field == "WPA") ||
          (field == "WPA2")) {
        authBase = field;
      }
      else if ((field == "EAP") ||
               (field == "PSK")) {
        authentications[authIdx++] = authBase "-" field;
        dbg("      authentication: " authentications[authIdx-1]);
      }
      else if ((field == "CCMP") ||
               (field == "TKIP")) {
        encryptions[encIdx++] = field;
        dbg("      encryption    : " field);
      }
      else {
        dbg("      unknown field: " field);
      }
    }

    # Save the security options for this flag.
    # There is one per permutation of authentication and encryption.
    dbg("    security:");
    for (i = 1; i < authIdx; ++i) {
      for (j = 1; j < encIdx; ++j) {
        gotSecurity = 1;
        security[secIdx++] = authentications[i] "." encryptions[j];
        dbg("      " security[secIdx-1]);
      }
    }
  }

  # Only process ESS (AP) records, ignore IBSS (ad-hoc)
  if (isAp != 1) {
    dbg("    ignoring as not an AP");
    next;
  }

  # If no security flags were detected use the default security of open/none
  if (gotSecurity == 0) {
    security[secIdx++] = "OPEN.NONE";
    dbg("    No security flags:");
    dbg("      " security[secIdx-1]);
  }

  # Get any optional ssid, trim any leading space
  ssid = "";
  {
    skipRecordSpace();
    res = match(record, "^[[:print:]]+");
    if (res != 0) {
      ssid = substr(record, RSTART, RLENGTH);
    }
    dbg("  ssid          : " ssid);
  }

  # Construct an output record for each security option.
  for (i = 1; i < secIdx; ++i) {

    # Do the conversions
    vals[];
    split(security[i], vals, ".");
    authentication = convertAuthentication(vals[1]);
    encryption     = convertEncryption(vals[2]);
    channel        = freq2Channel(settings["frequency"]);
    if ((authentication == "") || (encryption == "") || (channel == "")) {
      dbg("Ignoring unknown value: " authentication " " encryption " " channel);
      continue;
    }

    # Note the item names hear should match those in wlan_sta_linux.js:updateScanInfo()
    lines[linesIdx++] =                                                 \
      "    {"                                                           \
      sprintQuotedPair("ssid"          , ssid                   , ",")  \
      sprintQuotedPair("bssid"         , settings["bssid"]      , ",")  \
      sprintQuotedPair("authentication", authentication         , ",")  \
      sprintQuotedPair("encryption"    , encryption             , ",")  \
      sprintQuotedPair("channel"       , channel                , ",")  \
      sprintQuotedPair("signalLevel"   , settings["signalLevel"]     )  \
      "  }"
    dbg(lines[linesIdx-1]);
  }
}

#
# Functions to support "iw dev wlan_staN" output below:
#

function getBssid() {
  # Get the bssid
  if (match($0, "([[:xdigit:]]{2}:){5}[[:xdigit:]]{2}") != 0) {
    bssid = substr($0, RSTART, RLENGTH);
    dbg("  bssid      : " bssid);

    # See if we are connected to this AP
    if (match($0, "-- associated") != 0) {
      connectedBssid = bssid;
      dbg("    connected to this AP");
    }
  }
  else {
    dbg("bssid error");
    exit 1;
  }
}

function getMandatoryFields() {
  # Look for the other fields
  frequency   = "";
  signalLevel = "";
  ssid        = "";
  gotSsid     = "";
  while ((frequency   == "") ||
         (signalLevel == "") ||
         (gotSsid     == "") ||
         (isAp        == "")) {
    if (match($0, "freq:[[:space:]]+[[:digit:]]+") != 0) {
      match($0, "[[:digit:]]+");
      frequency = substr($0, RSTART, RLENGTH);
      dbg("  frequency  : " frequency);
    }
    else if (match($0, "capability:") != 0) {
      if (match($0, "ESS") != 0) {
        isAp = 1;
        dbg("  type       : AP");
      }
      else {
        isAp = 0;
        dbg("  type       : not an AP");
      }
      if (match($0, "Privacy") != 0) {
        protected = "yes";
        dbg("  protected  : yes");
      }
    }
    else if (match($0, "signal:[[:space:]]+[+-]*[[:digit:]]+.[[:digit:]]+") != 0) {
      match($0, "[+-]*[[:digit:]]+.[[:digit:]]+");
      signalLevel = substr($0, RSTART, RLENGTH);
      dbg("  signalLevel: " signalLevel);
    }
    else if (match($0, "SSID:[[:space:]]+") != 0) {
      gotSsid = "yes";
      if (match($0, ":[[:space:]]+[[:print:]]+") != 0) {
        ssid = substr($0, index($0, ":") +2, RLENGTH);
      }
      dbg("  ssid       : " ssid);
    }
    stillLines = getline;
  }
}

function getSecurityBlock(authBase) {
  authentications[];
  encryptions[];
  authIdx = 1;
  encIdx = 1;

  # Skip the version line
  stillLines = getline;

  # Process until the end of the block is detected
  #   - in block  :              * <text>
  #   - past block:     <text>:  * <text>
  while (stillLines == 1) {
    if (match($0, "^[[:space:]]+\*[[:space:]]+[[:print:]]+$") == 0) {
      break;
    }
    if (match($0, "* Pairwise ciphers:") != 0) {
      if (match($0, "CCMP") != 0) {
        encryptions[encIdx++] = "CCMP";
        dbg("    encryption: " encryptions[encIdx -1]);
      }
      if (match($0, "TKIP") != 0) {
        encryptions[encIdx++] = "TKIP";
        dbg("    encryption: " encryptions[encIdx -1]);
      }
    }
    else if (match($0, "* Authentication suites:") != 0) {
      if (match($0, "IEEE 802.1X") != 0) {
        authentications[authIdx++] = authBase "-EAP";
        dbg("    authentication: " authentications[authIdx -1]);
      }
      if (match($0, "PSK") != 0) {
        authentications[authIdx++] = authBase "-PSK";
        dbg("    authentication: " authentications[authIdx -1]);
      }
    }
    stillLines = getline;
  }

  # Add the supported security combinations
  # There is one per permutation of authentication and encryption.
  dbg("    security:");
  for (i = 1; i < authIdx; ++i) {
    for (j = 1; j < encIdx; ++j) {
      gotSecurity = 1;
      security[secIdx++] = authentications[i] "." encryptions[j];
      dbg("      " security[secIdx-1]);
    }
  }
}

# Construct an output record for each security option.
function createSecurityRecords() {
  for (i = 1; i < secIdx; ++i) {

    # Do the conversions
    vals[];
    split(security[i], vals, ".");
    authentication = convertAuthentication(vals[1]);
    encryption     = convertEncryption(vals[2]);
    channel        = freq2Channel(frequency);

    # Note the item names hear should match those in wlan_sta_linux.js:updateScanInfo()
    lines[linesIdx++] =                                        \
      "    {"                                                  \
      sprintQuotedPair("ssid"          , ssid          , ",")  \
      sprintQuotedPair("bssid"         , bssid         , ",")  \
      sprintQuotedPair("channel"       , channel       , ",")  \
      sprintQuotedPair("signalLevel"   , signalLevel   , ",")  \
      sprintQuotedPair("authentication", authentication, ",")  \
      sprintQuotedPair("encryption"    , encryption         )  \
      "  }";
    dbg(lines[linesIdx-1]);
  }
}

function flushAnyPendingSecurityRecords() {
  if (protected == "yes") {
    if (secIdx == 1) {
      # No security blocks found so must be WEP
      dbg("  Protected but no security blocks, security:");
      security[secIdx++] = "WEP.TKIP";
      dbg("      " security[secIdx-1]);
    }

    createSecurityRecords();
  }
}

# Process the remaining blocks, return when end of file.
function doBlocks() {
  protected = "";
  connectedBssid = "";
  stillLines = 1;
  while (stillLines == 1) {

    # Look for start of a BSS block
    if (match($0, "^BSS") != 0) {
      dbg($0);

      flushAnyPendingSecurityRecords();

      # Reset state for new block
      protected = "";
      isAp = "";
      security[];
      secIdx = 1;

      getBssid();

      stillLines = getline;
      getMandatoryFields();

      # Only process ESS (AP) blocks, ignore IBSS (ad-hoc)
      if (isAp != 1) {
        dbg("    ignoring as not an AP");
        stillLines = getline;
        continue;
      }

      if (protected != "yes") {
        # Use the default security of open/none
        dbg("  Not protected, security:");
        security[secIdx++] = "OPEN.NONE";
        dbg("      " security[secIdx-1]);

        createSecurityRecords();
        stillLines = getline;
        continue;
      }
    }

    if (protected == "yes") {
      # Look for the optional blocks and work out the authentication
      # and encryption combinations supported.

      # Is it a security block ?
      authBase = "";
      if (match($0, "RSN:") != 0) {
        authBase = "WPA2";
      }
      else if (match($0, "WPA:") != 0) {
        authBase = "WPA";
      }
      if (authBase != "" ) {
        # Yes, parse it
        dbg("  Security block: " authBase);
        getSecurityBlock(authBase);
      }
      else {
        # No, read next line
        stillLines = getline;
      }
    }
    else {
      # Not protected, look for the next BSS block
      stillLines = getline;
    }
  }

  flushAnyPendingSecurityRecords();
}

# Process iw dev wlan_staN output
/^BSS[[:space:]]+([[:xdigit:]]{2}:){5}[[:xdigit:]]{2}/ {
  writeConnected = 1;
  doBlocks();
}

END {
  # Construct valid JSON data for the scan array
  printf("  \"apConfigs\":[\n");
  for (i = 1; i < linesIdx-1; ++i) {
    printf("%s,\n", lines[i]);
  }
  printf("%s\n", lines[i]);

  if (writeConnected == 1) {
    print "  ],"
    print " " sprintQuotedPair("connectedBssid", connectedBssid);
  }
  else {
    print "  ]"
  }
  exit 0;
}
